---
name: async-await-coder
description: Detects and fixes missing async and await keywords in JavaScript or TypeScript code. Use this skill when reviewing code involving Promises, asynchronous API calls, or when debugging "Promise <pending>" issues and syntax errors related to asynchronous execution.
---
# Fix Async/Await Keywords

This skill ensures that asynchronous logic in JavaScript and TypeScript is correctly implemented by identifying where `async` or `await` keywords are missing.

## Instructions

1. **Identify Asynchronous Calls**: Search the codebase for functions that return a `Promise` (e.g., fetch, database queries, file system operations, or functions explicitly marked `async`).
2. **Verify Await Usage**: For every identified asynchronous call, check if it is preceded by the `await` keyword.
* If `await` is missing and the result is treated as a resolved value, insert `await`.
* If the call is inside a loop, suggest using `Promise.all()` for parallel execution or ensure `await` is used correctly for sequential execution.


3. **Ensure Async Context**: When an `await` keyword is added or found, verify that the containing function is marked with the `async` keyword.
* If the parent function is missing `async`, add it to the function declaration or arrow function definition.


4. **Check Callback Functions**: Specifically look at array methods like `.map()`, `.forEach()`, and `.filter()`.
* Ensure that if an asynchronous callback is passed, the callback itself is marked `async`.
* Advise the user that `.filter()` and `.forEach()` do not wait for asynchronous callbacks and suggest alternatives like `for...of` if sequential execution is required.


5. **Error Handling**: When fixing async/await structures, ensure the code is wrapped in a `try...catch` block if proper error handling is not already present.

## Examples

### Missing await

* **Before**: `const data = fetchData();`
* **After**: `const data = await fetchData();`

### Missing async

* **Before**: `function process() { await doSomething(); }`
* **After**: `async function process() { await doSomething(); }`

### Incorrect loop handling

* **Before**: `items.forEach(async (item) => { await save(item); });`
* **After**: `for (const item of items) { await save(item); }` (for sequential) OR `await Promise.all(items.map(async (item) => await save(item)));` (for parallel)

## Best Practices

* Prefer `async/await` syntax over raw `.then()` chains for readability.
* Always check that top-level await is supported by the environment before suggesting it outside of a function.

Would you like me to apply this Skill to a specific file in your project or search for async issues across your workspace?