---
name: node-debug-logging-middleware
description: Adds a plug-and-play Node.js middleware that instruments Express and Koa apps to emit detailed debugging logs (stack traces, request context, intermediate values) to a configurable log file path. Triggers: error/exception handling, request lifecycle boundaries, manual instrumentation points.
---

Skill purpose

This Skill provides a clear, repeatable way to add comprehensive debugging logs to Node.js backends (focused on Express and Koa). It modifies or supplies middleware and small helper functions that capture and persist:
- request context (headers, url, method, client IP)
- timestamps for lifecycle stages
- intermediate values at instrumentation points
- full stack traces for errors and unhandled rejections
- correlation IDs to trace a single request across async boundaries

The logs are written to a user-configurable file path in JSON lines format for easy ingestion.

Step-by-step instructions Claude must follow

1. Ask the user for the target framework (Express or Koa), desired log file path, and preferred log level threshold (e.g., debug/info/warn/error). If any detail is missing, default to Express, ./logs/debug.log, and "debug".
2. Generate or modify a middleware file that:
   a. Attaches a correlation ID to each incoming request (generate UUIDv4 if absent).
   b. Records entry timestamp and basic request context.
   c. Wraps request lifecycle to capture intermediate instrumentation points via helper functions (traceValue(ctx, label, value)).
   d. On response finish, records response status, timing, and writes a compact JSON line representing the request lifecycle.
   e. Catches synchronous exceptions and delegates to an error logger that writes stack traces and context to the same log file.
   f. Binds to process events (uncaughtException, unhandledRejection) to log full stack traces and current active request contexts when possible.
3. For Express: produce an Express middleware function (req, res, next) plus helpers. For Koa: produce equivalent middleware (async (ctx, next) => { ... }).
4. Provide small, copy-paste usage examples showing how to mount the middleware early in the app and how to call traceValue at key points in route handlers or services.
5. When producing code, ensure the log file path is configurable via environment variable (DEBUG_LOG_PATH) and default to ./logs/debug.log. Use safe file append (fs.createWriteStream with { flags: 'a' }) and handle write errors gracefully (fallback to console.error).
6. Recommend log rotation or external log management for production (note in Best Practices). Include instructions for running a quick manual test (sample request, intentional error) to verify logs are produced.
7. If user requests, adapt code to their existing repository structure or provide step-by-step patch instructions.

Usage examples

- Example 1 (Express): mount the middleware as the first app.use call, optionally set process.env.DEBUG_LOG_PATH, and instrument key logic with traceValue(req, 'dbResult', dbResult).
- Example 2 (Koa): use the provided async middleware, set the env var, and call ctx.state.trace('afterValidation', someValue) in handlers.

Best Practices

- Keep this middleware early in the middleware chain so it captures original request context.
- Avoid logging sensitive fields (authorization tokens, passwords). Provide a sanitizer hook to redact fields before writing.
- Use correlation IDs to connect logs from async operations and external services.
- Enable log rotation (logrotate) or forward logs to a centralized system in production; large JSON logs can grow quickly at debug level.
- Prefer structured queries on the JSON lines output rather than grepping raw text.

Related files

- examples/debug-middleware.js — Express and Koa middleware + helpers (provided below)
- examples/usage-express.js — minimal Express app demonstrating installation
- examples/usage-koa.js — minimal Koa app demonstrating installation

