'use strict';

// Lightweight debug logging middleware for Express and Koa
// Writes JSON lines to DEBUG_LOG_PATH (default ./logs/debug.log)

const fs = require('fs');
const path = require('path');
const { v4: uuidv4 } = require('uuid');

const DEFAULT_PATH = path.resolve(process.env.DEBUG_LOG_PATH || './logs/debug.log');

function ensureDirForFile(filePath) {
  const dir = path.dirname(filePath);
  if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });
}

ensureDirForFile(DEFAULT_PATH);

let writeStream;
function getWriteStream(filePath = DEFAULT_PATH) {
  if (writeStream) return writeStream;
  try {
    writeStream = fs.createWriteStream(filePath, { flags: 'a' });
    writeStream.on('error', (err) => {
      console.error('[debug-logger] write error:', err);
      writeStream = null;
    });
  } catch (e) {
    console.error('[debug-logger] failed to open log file:', e);
  }
  return writeStream;
}

function safeStringify(obj) {
  try {
    return JSON.stringify(obj);
  } catch (e) {
    return '"[unserializable]"';
  }
}

// Basic sanitizer hook - remove common sensitive keys
function defaultSanitizer(obj) {
  if (!obj || typeof obj !== 'object') return obj;
  const clone = Array.isArray(obj) ? obj.slice() : Object.assign({}, obj);
  const sensitive = ['authorization', 'password', 'passwd', 'token'];
  for (const k of Object.keys(clone)) {
    if (sensitive.includes(k.toLowerCase())) clone[k] = '[REDACTED]';
  }
  return clone;
}

function createLogger({ logPath, sanitizer } = {}) {
  const stream = getWriteStream(logPath || DEFAULT_PATH);
  const sanitize = sanitizer || defaultSanitizer;

  function write(entry) {
    if (!stream) return console.error('[debug-logger] no stream, entry:', entry);
    const line = safeStringify(entry);
    stream.write(line + '\n');
  }

  function logRequestLifecycle(meta) {
    write({ type: 'request', timestamp: Date.now(), ...meta });
  }

  function logError(meta) {
    write({ type: 'error', timestamp: Date.now(), ...meta });
  }

  return { logRequestLifecycle, logError, sanitize };
}

// Express middleware
function expressMiddleware(opts = {}) {
  const logger = createLogger(opts);

  return function (req, res, next) {
    try {
      req._debug = req._debug || {};
      req._debug.id = req.headers['x-correlation-id'] || uuidv4();
      req._debug.start = Date.now();
      req._debug.traces = [];

      function trace(label, value) {
        req._debug.traces.push({ label, value, ts: Date.now() });
      }

      req.debugTrace = trace; // expose helper to app code

      // on response finish
      res.on('finish', () => {
        const entry = {
          id: req._debug.id,
          method: req.method,
          url: req.originalUrl || req.url,
          status: res.statusCode,
          start: req._debug.start,
          duration: Date.now() - req._debug.start,
          clientIp: req.ip || req.connection && req.connection.remoteAddress,
          traces: logger.sanitize(req._debug.traces),
          requestHeaders: logger.sanitize(req.headers)
        };
        logger.logRequestLifecycle(entry);
      });

      next();
    } catch (err) {
      logger.logError({ message: 'middleware failure', error: (err && err.stack) || String(err) });
      next(err);
    }
  };
}

// Express error handler (final error middleware)
function expressErrorHandler(opts = {}) {
  const logger = createLogger(opts);
  return function (err, req, res, next) {
    try {
      const entry = {
        id: (req && req._debug && req._debug.id) || uuidv4(),
        message: err && err.message,
        stack: err && err.stack,
        method: req && req.method,
        url: req && (req.originalUrl || req.url),
        requestHeaders: req && logger.sanitize(req.headers),
        traces: req && req._debug && logger.sanitize(req._debug.traces)
      };
      logger.logError(entry);
    } catch (e) {
      console.error('[debug-logger] failed to log error:', e);
    }
    next(err);
  };
}

// Koa middleware
function koaMiddleware(opts = {}) {
  const logger = createLogger(opts);

  return async function (ctx, next) {
    ctx.state._debug = ctx.state._debug || {};
    ctx.state._debug.id = ctx.headers['x-correlation-id'] || uuidv4();
    ctx.state._debug.start = Date.now();
    ctx.state._debug.traces = [];

    ctx.state.trace = function (label, value) {
      ctx.state._debug.traces.push({ label, value, ts: Date.now() });
    };

    try {
      await next();
    } catch (err) {
      const entry = {
        id: ctx.state._debug.id,
        message: err && err.message,
        stack: err && err.stack,
        method: ctx.method,
        url: ctx.url,
        requestHeaders: logger.sanitize(ctx.headers),
        traces: logger.sanitize(ctx.state._debug.traces)
      };
      logger.logError(entry);
      throw err;
    } finally {
      const entry = {
        id: ctx.state._debug.id,
        method: ctx.method,
        url: ctx.url,
        status: ctx.status,
        start: ctx.state._debug.start,
        duration: Date.now() - ctx.state._debug.start,
        clientIp: ctx.ip,
        traces: logger.sanitize(ctx.state._debug.traces),
        requestHeaders: logger.sanitize(ctx.headers)
      };
      logger.logRequestLifecycle(entry);
    }
  };
}

// Global process handlers
function bindProcessHandlers(opts = {}) {
  const logger = createLogger(opts);
  process.on('uncaughtException', (err) => {
    logger.logError({ message: 'uncaughtException', stack: err && err.stack });
    console.error(err);
    // do not exit here; leave to process manager
  });

  process.on('unhandledRejection', (reason) => {
    logger.logError({ message: 'unhandledRejection', reason: (reason && reason.stack) || reason });
    console.error('unhandledRejection', reason);
  });
}

module.exports = {
  expressMiddleware,
  expressErrorHandler,
  koaMiddleware,
  bindProcessHandlers
};
