---
name: flutter-korean-one-line-docs
description: Adds or updates one-line Korean summary comments for every function across a Flutter (Dart) project. Triggered when a user asks to document all functions, generate missing short Korean summaries, or normalize function comment style across the repo.
---

# Skill: Flutter Korean One-Line Docs

Purpose

This Skill scans a Flutter (Dart) project and adds or updates a single-line Korean summary comment for every function (top-level functions, class methods, getters, setters, and constructors) across all files and access levels. The summary is one short Korean sentence describing what the function does. Designed for teams who want consistent, concise Korean-line documentation across the codebase.

Step-by-step instructions (what Claude must do)

1. Identify project root and all Dart source files.
   - Look for files with the .dart extension under the given directory (recursively), excluding common build/artifacts directories (e.g., build/, .dart_tool/, .git/).
2. For each Dart file, parse or analyze the file to locate function-like declarations including:
   - Top-level functions
   - Class methods (including static)
   - Getters and setters
   - Constructors (named and unnamed)
3. For each function-like declaration, determine whether a one-line Korean summary comment already exists immediately above the declaration.
   - Acceptable existing comment formats: // <text> or /// <text> (single-line). If the existing comment is longer than one sentence or is in a language other than Korean, treat it as needing replacement only if user requested replacement; otherwise only add comments where none exist.
4. Generate a one-line Korean summary for the function. Requirements for the generated summary:
   - Use Korean language exclusively.
   - Produce a single concise sentence (one main clause), no more than ~15–20 Korean words when possible.
   - Start with the function's intent (action) and include the subject when helpful (e.g., "주어진 문자열에서 공백을 제거합니다.").
   - Avoid parameter-level details. If the function name already conveys parameters, paraphrase function intent plainly.
   - For constructors: describe what the constructor configures or represents (e.g., "기본 위젯 설정으로 새 인스턴스를 생성합니다.").
5. Insert or update the comment directly above the declaration using the single-line comment syntax.
   - Prefer '///' (Dart doc single-line) when adding documentation comments for public APIs; use '//' if the function is private (starts with underscore) and user preferences indicate non-doc comments. If user hasn't specified preference, default to '///' for all functions.
   - Ensure exactly one comment line above the declaration (replace multiple existing lines with the single generated line if user requested normalization; otherwise only add if missing).
6. Preserve file formatting and import statements. Do not change other code content. Maintain existing blank lines above declarations when possible.
7. Create a draft or preview of changes as a patch or diff per file (e.g., unified diff or file-modified content) rather than committing directly, unless user requests applying changes.
8. Provide a summary report listing:
   - Number of files scanned
   - Number of functions found
   - Number of comments added
   - Number of comments updated (if any)
   - Any parsing failures or ambiguous constructs that require manual review
9. When asked, apply the changes to files and optionally create a branch/commit (if repository access and commit instructions are provided). Otherwise return the generated diffs and summary.

Usage examples

- User prompt: "Document all functions in this Flutter project with one-line Korean summaries."
  - Response: scan the repo, produce diffs adding '///' one-line Korean sentences above each function that lacks them, and provide the summary report.

- User prompt: "Only add comments for functions that don't have any comment yet. Do not replace existing comments."
  - Response: add Korean one-line comments only where absent, leaving existing comments untouched.

- User prompt: "Normalize comments: replace all existing function comments with concise one-line Korean summaries."
  - Response: replace existing single-line comments with newly generated Korean one-line summaries and report counts of replaced comments.

Examples of generated comments (Korean sentences):

- For a function trimSpaces(String s):
  /// 문자열의 앞뒤 공백을 제거합니다.

- For a class method calculateTotal():
  /// 장바구니의 총 금액을 계산합니다.

- For a private helper _parseDate(String s):
  /// 문자열을 DateTime으로 변환합니다.

- For a constructor MyWidget():
  /// 기본 설정으로 MyWidget 인스턴스를 생성합니다.

Best practices

- Ask the user whether they prefer '///' (Dart doc) or '//' for private functions before modifying files. Default to '///' when unspecified.
- Provide diffs first and request approval before altering repository files or committing changes.
- For ambiguous or very complex functions (one-line summary would be misleading), flag them for manual review rather than generating a vague comment.
- Keep summaries objective and avoid implementation details or parameter lists.
- Respect existing file linting and formatting; run dart format externally if the user requests post-processing.

When to ask follow-up questions

- When the user hasn't specified whether to replace existing comments: ask whether to preserve, replace, or only add missing comments.
- When repository access or file list is not provided: ask for the project root or upload of files.
- When encountering non-Dart files or generated code: ask whether to skip those directories.

Related files or scripts

- No additional supporting scripts are required by default. If requested, the Skill can provide a small example script (e.g., a Python or Node.js tool) to produce diffs automatically. Request this if needed.
