---
name: code-cleanup-refactor
description: Clean up and refactor source code to improve readability and naming while preserving behavior. Triggered by requests to reformat, rename variables/functions for clarity, and apply consistent style across common languages (Python, JavaScript, Java, C, C++). Often used when a user asks to "clean up code", "improve readability", or "refactor names".
---

# Skill: Code Cleanup & Refactor

Purpose

This Skill reformats and refactors source code to improve readability and maintain behavior. It supports common languages (Python, JavaScript/TypeScript, Java, C, C++) and performs:
- consistent indentation and whitespace
- line-wrapping and blank-line insertion for clarity
- renaming variables, functions, and simple identifiers to more descriptive, consistent names
- minimal structural refactors (extract small helper functions, simplify complex expressions) only when safe
- producing an edit-ready patch or direct edits for use by a code-editing agent/tool

Step-by-step instructions Claude must follow

1. Identify language and scope
   - Detect the language (Python, JavaScript/TypeScript, Java, C, C++) from file extension or code heuristics.
   - Confirm the requested scope: single file, multiple files, or selected region. If scope is ambiguous, ask a clarifying question.
2. Preserve behavior
   - Prioritize preserving program semantics. Do not change algorithm logic, public API signatures, or external behavior unless user explicitly permits API changes.
   - For renames, ensure consistent renaming across the provided scope and update all references. Prefer local/safe renames over global changes when risk is unclear.
3. Format and whitespace
   - Apply idiomatic formatting for the detected language (PEP8-like for Python, Prettier-like conventions for JS/TS, common Java/C/C++ indentation and brace style).
   - Fix indentation, trailing whitespace, consistent line breaks, and blank lines between logical blocks.
4. Improve names
   - Suggest and apply more descriptive names for variables and functions: prefer nouns for variables, verbs for functions, and consistent casing per language conventions (snake_case for Python, camelCase for JS/Java/C/C++ unless otherwise specified).
   - Avoid overly long names; keep names concise but descriptive. When renaming, include rationale in comments or in an accompanying changelog note.
5. Small safe refactors
   - Where readability suffers, perform minimal refactors: extract repeated expressions into a well-named helper, simplify nested conditionals, or inline trivial variables. Only make these changes if they do not alter behavior and are straightforward to review.
6. Produce outputs suitable for editing agents
   - Provide either:
     - A unified diff/patch (git-style) describing changes, or
     - A set of file edits with before/after contents, or
     - Directly edit files when the caller's tooling supports in-place edits.
   - Include a short summary of changes, list of renamed identifiers and their mappings, and any reasons for nontrivial refactors.
7. Validation
   - Run lightweight static checks where possible (linting rules or formatting tools) to verify formatting. Do not run code or tests unless explicitly authorized.
8. Clarify and iterate
   - If a suggested rename or refactor could be controversial (public APIs, ambiguous intent), present alternatives and ask for confirmation before applying global changes.

Usage examples

Example 1 — Single-file cleanup (Python)
- User supplies a Python file and asks: "Make this file more readable and improve names." The Skill:
  - Detects Python, runs a formatter pass (PEP8 conventions), proposes renames (e.g., tmp -> user_count), updates all references, returns a unified diff and a changelog mapping names.

Example 2 — JS/TS region refactor
- User highlights a function in a JavaScript file and requests naming improvements only. The Skill:
  - Applies camelCase naming, updates references within the file/scope, and returns the edited region and a short rationale.

Example 3 — Multi-file safe renaming (C++)
- User requests renaming an internal helper function used across a module. The Skill:
  - Confirms scope (single module), ensures all translation-unit references are updated, produces a patch, and lists replaced symbols.

Best practices

- Always ask quick clarifying questions for ambiguous scopes or when changes might affect public APIs.
- Keep rename mappings explicit and reversible: provide original -> new name lists.
- Favor language-idiomatic formatting and naming conventions unless the user specifies a different style guide.
- Limit automated structural changes to small, easily-reviewable edits.
- When in doubt, provide suggestions and let the user approve automated edits.

When to ask clarifying questions

- Unclear language or mixed-language snippets
- Scope not specified (whole repo vs single file vs selection)
- Potential API/public-interface changes
- When renaming symbols that might be externally referenced (e.g., exported functions)

Related files and tooling

- If available, reference project style configs (e.g., .prettierrc, pyproject.toml, .clang-format) and honor them when formatting. If no configs are provided, use reasonable defaults described above.

