---
name: js-refactor-cleanup-skill
description: JavaScript-focused refactor and cleanup Skill. Triggered by requests to clean up, rename, or refactor JavaScript/TypeScript code. Includes example JS files illustrating common refactor cases; the Skill's instructions reference these examples for guidance and tests.
---
# Skill: JavaScript Cleanup & Refactor (JS-focused)

Purpose

This Skill specializes in refactoring and cleaning up JavaScript and TypeScript code to improve readability, naming, and structure while preserving behavior. It includes prepared example files demonstrating common refactor patterns (naming, async/await conversion, modularization) which Claude should reference when proposing or performing edits.

Step-by-step instructions Claude must follow

1. Identify language and scope
   - Confirm the code is JavaScript or TypeScript. If mixed languages are present, ask the user whether to proceed focusing only on JS/TS.
   - Confirm scope: single file, multiple files, selection, or repo. If unspecified, ask.
2. Reference example cases
   - Use the included example files in ./examples/ as reference implementations and pattern examples. When the user asks for a refactor that matches an example pattern (naming cleanup, async refactor, module export cleanup), mention which example(s) were used as guidance and why.
   - Example files to reference:
     - ./examples/simple-refactor.js — basic naming and small helper extraction.
     - ./examples/async-refactor.js — callback-to-async/await and clearer error handling.
     - ./examples/module-refactor.js — clearer module exports and consistent naming.
3. Preserve behavior
   - Never change program semantics, public API, or exported function signatures unless the user explicitly permits API changes.
   - Run renames consistently across the provided scope. If a rename might affect external callers (exports/imports), flag it and ask for confirmation.
4. Formatting and idiomatic style
   - Apply Prettier-like conventions (2-space indentation, semicolons per project style if present, consistent single/double quotes as configured).
   - Prefer camelCase for variables and functions, PascalCase for classes, and consistent naming for constants (UPPER_SNAKE_CASE) unless the project style indicates otherwise.
   - Honor project config files if present (.prettierrc, eslint, tsconfig.json). If not present, default to common JS conventions described above.
5. Small safe refactors
   - Perform minimal, easily-reviewable structural changes: extract small helpers, collapse duplicated code into functions, convert promise .then chains to async/await where it improves clarity and preserves behavior, and replace verbose loops with array helpers when semantics are unchanged.
   - For each nontrivial change, include a brief rationale and the mapping of renamed identifiers.
6. Produce outputs suitable for editing agents
   - Provide a unified diff/patch, or before/after file contents, and a concise changelog listing renamed identifiers and a one-line reason for each change.
   - If edits reference the example files for patterns, include a note like: "Refactor guided by ./examples/async-refactor.js" and link the example path.
7. Validation
   - Run lightweight static checks where possible (lint rule inferences); do not execute code or run tests unless explicitly authorized.
8. Clarify and iterate
   - Ask clarifying questions for ambiguous scope, potential API changes, or when multiple safe refactor alternatives exist.

Usage examples

Example 1 — Single-file naming cleanup
- User supplies a JS file and asks: "Make names clearer and format this file." The Skill:
  - Detects JS, applies Prettier-like formatting, suggests concise descriptive variable/function renames, updates references in the file, and returns a unified diff plus a mapping (original -> new).
  - If the rename affects exports, the Skill notes which imports outside scope may need updates and asks for confirmation before a global rename.

Example 2 — Convert callbacks/promises to async/await
- User asks to modernize a function that uses callbacks or chained .then calls. The Skill:
  - Uses ./examples/async-refactor.js as a pattern for converting to async/await and improving error handling, applies the conversion where behavior is preserved, and returns the edited code with a summary of changes.

Example 3 — Modularization and export cleanup
- User wants clearer module boundaries and consistent exports. The Skill:
  - References ./examples/module-refactor.js for patterns, consolidates related functions into a module or object where appropriate, preserves public export names or requests confirmation for changes.

Best practices

- Always confirm scope and whether exported names can change.
- Provide explicit rename mappings and short rationales for structural changes.
- Use the example files in ./examples/ as guidance and mention them when used.
- Keep structural changes minimal and easily reversible; prefer suggestions over wholesale transformations without user sign-off.

When to ask clarifying questions

- Unclear scope (single file vs repo-wide).
- Potential public API or export/import changes.
- Mixed-language or TypeScript-specific typing changes.
- If the requested refactor matches an example pattern but the user expects a different style.

Related files and tooling

- The Skill honors project style configs (.prettierrc, .eslintrc, tsconfig.json) if present. If none are provided, follow the defaults above.
- Reference example files in ./examples/ for concrete refactor patterns.

Links to example files

- ./examples/simple-refactor.js
- ./examples/async-refactor.js
- ./examples/module-refactor.js
