---
name: unused-code-refactor-suggester
description: Scans project source files to identify functions and classes that are not referenced elsewhere within the same file (file-scoped unused code). For each finding, generates a concise refactoring_report.md entry recommending deletion or preservation with a short reason and risk summary. Common triggers: "find unused in file", "identify dead code per-file", "generate refactoring report".
---

Skill purpose

This Skill examines source files and identifies functions, classes, and top-level definitions that are not referenced anywhere else in the same file (file-scoped unused items). For each item found, the Skill creates or updates a refactoring_report.md file. Each report entry contains a concise recommendation (Delete or Keep), a one-line reason for deletion, and a one-line summary of risk if deleted.

Step-by-step instructions Claude must follow

1. Determine target files: accept a file list from the user or scan the repository for supported source file extensions. By default consider: .py, .js, .ts, .jsx, .tsx, .java, .go, .rb. If the user supplies specific files or a directory, restrict scanning accordingly.
2. For each file, parse or analyze to extract top-level and module-level definitions: functions, classes, and exported symbols. Prefer language-aware parsing (AST) where available; if not available, use conservative regex heuristics to find definitions.
3. For each extracted definition in a file, search the same file (excluding comments and string literals if feasible) for references to the definition name:
   - Consider direct references (name, method call, instantiation) and attribute access (For classes, look for "new ClassName" or "ClassName(" or "ClassName.").
   - For languages with export/import syntax, do NOT treat references in other files as part of this Skill's scope—this Skill only flags items unused within the same file.
4. Determine unused status: if no references are found within the file (other than the definition itself), mark as "file-scoped unused."
5. Build report entries: for each file-scoped unused item produce a one-paragraph entry containing:
   - Item identifier: file path, line number or approximate location, definition name, type (function/class)
   - Recommendation: Delete or Keep
   - Deletion reason (one short sentence)
   - Deletion risk (one short sentence summarizing potential breakage or false-positive concern)

   Recommendation heuristics (defaults):
   - Recommend Delete if:
     - The definition is not referenced within the file, and
     - The name is not exported by obvious export syntax in the same file (e.g., module.exports, export keyword) and
     - The name does not match a common entrypoint pattern (e.g., main, CLI handlers) and
     - The definition is not decorated/annotated with obvious usage markers (e.g., @public, // used externally) when detectable.
   - Recommend Keep if:
     - The definition is exported or annotated as public, OR
     - The name matches common patterns that may be referenced externally (e.g., exported index, test hooks), OR
     - Detection is uncertain (report as Keep with note: "uncertain — possibly used externally").
6. Format and write refactoring_report.md:
   - If refactoring_report.md does not exist at repository root, create it. If it exists, append a clearly dated section for this run.
   - For each finding add a concise bullet or subheading with the four required pieces (identifier, Recommendation, Reason, Risk). Keep each reason/risk to one short sentence (1–2 lines).
   - Example entry format (concise):
     - File: src/utils/helpers.py:42 — function compute_format
       - Recommendation: Delete
       - Reason: No references in this file; appears to be dead helper logic.
       - Risk: May be used by other modules via dynamic import; verify cross-file usage before deletion.
7. Return to the user a summary: number of files scanned, number of file-scoped unused items, and path to refactoring_report.md. Offer next actions: run cross-file analysis, open patch/PR to remove items, or run tests.

Usage examples

- User: "Scan src/ for unused functions in each file and produce a report." Skill behavior: scan supported files in src/, produce refactoring_report.md with entries per unused item, and report counts.

- User: "Check file app/controllers/user_controller.rb for file-scoped unused classes and list recommendations." Skill behavior: analyze that single file, add matching entries to refactoring_report.md, and return the per-file findings.

Best practices

- When AST parsing is available for a language, prefer it over regex heuristics to reduce false positives.
- Be conservative: when unsure, recommend Keep with an uncertainty note rather than Delete.
- Always include the short risk sentence so human reviewers can judge deletion urgency.
- Mention export annotations or common framework hooks (e.g., Django signals, Rails callbacks) as reasons to avoid automatic deletion.
- Encourage follow-up: cross-file usage analysis and running tests before accepting deletions.

Placeholders and assumptions

- This Skill focuses only on file-scoped unused detection. It intentionally does NOT analyze cross-file references, package exports, or runtime/dynamic references.
- Supported file extensions are a reasonable default list; prompt the user if they want additional languages.

Output artifacts

- refactoring_report.md at repository root (created or appended). Each run adds a dated section and concise entries as described above.

Examples of report entries (short form)

- File: lib/math_ops.js:120 — function legacyAdd
  - Recommendation: Delete
  - Reason: No references in this file; helper appears unused.
  - Risk: Might be imported by other files via dynamic path; run cross-file scan before removal.

- File: src/models/order.py:30 — class OldOrderSerializer
  - Recommendation: Keep
  - Reason: Exported by module and likely used externally.
  - Risk: None immediate, but verify external callers before refactoring.

Related considerations

- If the user asks to proceed with deletions, advise creating a branch and opening a PR with each deletion change and the refactoring_report.md as justification.

