---
name: html-wireframe-nextjs
description: Generate an executable low-fidelity Next.js (App Router) wireframe app that runs in an Nx + pnpm monorepo. Triggered when a PM or designer requests an immediate HTML-previewable wireframe (2–4 page flow) with dummy data, basic interactions (modal, dropdown, toggle), loading/empty/error states, and navigation links. 
---

# HTML Wireframe (Next.js) Skill

Purpose

This Skill generates a ready-to-run, low-fidelity (lo-fi) Next.js wireframe app (App Router) that can be dropped into an Nx + pnpm monorepo. The output is minimal black/gray box UI showing layout and interaction only (no real API, DB, or design system). The wireframe includes: login/list/detail flow (2–4 pages), sections, lists, tables, forms, modal, dropdown, toggle, banners/alerts, loading/empty/error states, dummy data, and navigation links to verify user flow. The produced code must build and run with no external backend.

Step-by-step instructions (what Claude must do)

1. Confirm user intent and required pages/flow.
   - Ask minimal clarifying questions if needed: number of pages (2–4), primary components required (form, table, modal, etc.), and route names. If user already provided requirements, proceed.
2. Choose project layout and constraints.
   - Target: Next.js (App Router) app under an Nx workspace structure. Use simple file-based routes under app/ (e.g., app/(wireframe)/page.tsx, app/(wireframe)/login/page.tsx, app/(wireframe)/list/page.tsx, app/(wireframe)/detail/[id]/page.tsx).
   - Use only client components where interaction is needed ("use client").
   - Use plain CSS modules or inline styles (no Tailwind or design libs) to ensure minimal dependency footprint.
3. Create an executable minimal package structure and README instructions.
   - Provide package.json scripts (dev/build/start) and note pnpm usage. Keep external dependencies to a minimum (Next.js). For Nx monorepo, show placement guidance: apps/<wireframe-app>/...
4. Generate code files with clear content and comments.
   - Layout: a root layout that sets a neutral black/gray box aesthetic and basic header with navigation.
   - Pages: implement requested pages (login, list, detail, and optionally form) using lo-fi styles (gray boxes, simple typography). Each page should include:
     - Dummy data displayed in lists/tables/forms
     - Interactive elements: modal (confirm/cancel), dropdown, toggle, banner/alert
     - Loading, empty, and error states (simulate with query string or local state toggles)
     - Navigation links between pages to simulate user flow
   - Client components: small components for Modal, Dropdown, Toggle, Banner, Skeleton (loading), EmptyState, ErrorState, List/Table row.
5. Keep all data mocked locally.
   - No network calls, no DB, no Prisma. Provide a data file with example items and helper functions, e.g., libs/data/mock.ts.
6. Validate build/run instructions.
   - Provide clear commands to run within an Nx + pnpm monorepo (examples: pnpm install, pnpm --filter <app> dev or pnpm dev inside app folder). Explain where to place the app in an Nx repo (apps/<app-name>/).
7. Provide usage examples and showing how to toggle states.
   - Include examples of query strings or UI toggles to trigger loading/empty/error states and sample navigation flow steps.
8. Keep UI intentionally simple and neutral.
   - No branding, colors beyond grayscale, no pixel-perfect styling.
9. Deliver a ZIP-style file list (or code snippets) and a short README describing file placement and run steps.

Usage examples (prompts and outputs)

- Example user prompt to the Skill:
  "Create a 3-page wireframe: login -> list -> detail. Include modal, dropdown, toggle, banner, and simulated loading/empty/error states. Output code ready to run in an Nx + pnpm monorepo."

- Expected Skill output (summary):
  - File tree snippet and contents for a minimal Next.js app under apps/wireframe-app/
  - Key files: apps/wireframe-app/app/layout.tsx, app/page.tsx (home), app/login/page.tsx, app/list/page.tsx, app/detail/[id]/page.tsx, app/components/Modal.tsx, Dropdown.tsx, Toggle.tsx, Banner.tsx, mock-data.ts, globals.css, package.json, README.md
  - Short README with commands: pnpm install (root), pnpm --filter wireframe-app dev OR cd apps/wireframe-app && pnpm dev

Best practices

- Keep interactions client-only and stateful using React useState/useEffect.
- Use minimal, self-contained CSS to avoid collisions in monorepo.
- Provide clear comments in generated files to explain where to change dummy data or add routes.
- Expose simple toggles or query params to simulate loading/empty/error so reviewers can quickly verify states.
- Keep the wireframe limited to structure and flow; do not attempt to implement real APIs or business logic.

Suggested file locations and naming (example)

- apps/wireframe-app/
  - package.json
  - next.config.js
  - app/
    - layout.tsx
    - page.tsx (home or login)
    - login/page.tsx
    - list/page.tsx
    - detail/[id]/page.tsx
    - components/
      - Modal.tsx
      - Dropdown.tsx
      - Toggle.tsx
      - Banner.tsx
      - Skeleton.tsx
      - EmptyState.tsx
      - ErrorState.tsx
    - styles/globals.css
  - libs/mock-data.ts
  - README.md

When to ask follow-ups

- If the user did not specify page names, number of pages, or which interactive elements are required, ask those specific questions before generating code.

Related links and references

- Mention that output assumes Next.js App Router and Node >= 16, pnpm and Nx workspace. Provide inline README run commands and placement guidance in the generated README.

Examples of simple commands to run (to include in README)

- In monorepo root:
  - pnpm install
  - pnpm --filter wireframe-app dev

- Or inside the app folder:
  - cd apps/wireframe-app
  - pnpm install
  - pnpm dev

