---
name: developer-agent-guide
description: Implements a Claude Code Agent Skill that converts a planner's requirements and an HTML wireframe into a complete, runnable Nx monorepo (Next.js App Router frontend, NestJS Fastify backend, PostgreSQL + Prisma DB, Zod validation, and tests). Triggers: provide a planner spec + HTML wireframe, request full monorepo generation, or ask for incremental feature implementation.
---

# Developer Agent Guide (Summary)

Purpose

This Skill defines how the developer agent must convert a planner's specification and an HTML wireframe into a fully runnable Nx monorepo implementing only the planned features and preserving wireframe structure. The produced repository includes frontend (Next.js App Router), backend (NestJS + Fastify), Prisma schema + migration, Zod validation, and tests (Jest/Vitest/Playwright). The agent must produce code that builds and tests successfully and annotate unresolved ambiguities as TODOs and issues.

Step-by-step instructions Claude must follow

1. Input validation
   - Accept two inputs: (A) planner spec (requirements, acceptance criteria, AC/QA) and (B) HTML wireframe (structure-only). If either is missing, ask clarifying questions.

2. Scope enforcement
   - Implement only features explicitly defined in the planner spec. Do not add features not present in the spec.
   - Use the wireframe to derive page/component structure and layout only — do not treat visual details beyond structure as requirements.

3. Project scaffolding (Nx + pnpm)
   - Create an Nx workspace (pnpm) with logical apps/libs separation.
   - Apps: frontend-app (Next.js App Router) and api-app (NestJS configured with Fastify).
   - Libraries: shared/ui, shared/types, shared/db, shared/validation, feature-specific libs for modularization.

4. Wireframe → Page/component mapping
   - Parse the HTML wireframe to produce a page/component map: route paths, server vs client component decisions, and layout hierarchy.
   - Maintain the wireframe structure exactly for DOM/semantic structure; prefer Server Components for Next.js pages unless an interactive region requires client behavior.
   - For each page produce: route file (app/.../page.tsx), layout (if hierarchical), and presentational components under shared/ui or feature lib.

5. Backend design (NestJS Fastify)
   - Implement feature-aligned modules. Each feature becomes a Nest module with controllers, services, DTOs, and repository access.
   - Use Prisma client for DB access through a shared/db library and a provider in NestJS.
   - Expose REST endpoints matching frontend needs; document endpoints with OpenAPI (Swagger) so frontend can integrate.

6. Database modeling (Prisma)
   - Derive Prisma models strictly from planner AC/QA. Do not invent fields beyond what's necessary for AC.
   - Provide a complete Prisma schema and a migration plan (SQL migration file or Prisma migrate steps). If model details are ambiguous, add TODO comments and create a GitHub-style issue file in the repo describing the ambiguity.

7. Validation (Zod)
   - Add Zod schemas for input validation on both backend DTO parsing and frontend form handling. Keep schemas colocated in shared/validation or feature libs.

8. Tests (AC/QA-driven)
   - Write tests that verify acceptance criteria: unit tests (Jest or Vitest) for services/libs, integration tests for API routes, and end-to-end tests (Playwright) for key page flows.
   - Ensure test commands run via pnpm and pass locally in CI-like configuration.

9. Error handling & TODOs
   - For any ambiguous requirement or missing data, do not guess silently. Insert TODO comments in code, create a top-level /ISSUES.md listing the open questions, and add lightweight TODO tests that assert assumptions.

10. Deliverables and verification
   - Produce a runnable repo with scripts: pnpm install, pnpm build, pnpm test, pnpm migrate, pnpm dev
   - Include README with setup, migration, and how to run tests.
   - Ensure schema.prisma and at least one migration file are present.
   - Provide generated OpenAPI (Swagger) JSON accessible from api-app.

Usage examples

- Example 1: Full generation request
  - Input: planner.md (AC list) + wireframe.html
  - Command/Prompt to agent: "Generate a complete Nx monorepo implementing the planner ACs and mapping pages from wireframe.html. Provide repository as a zip and include migration and tests. Mark any ambiguous fields as TODO and add ISSUES.md."
  - Expected outcome: Complete repo scaffolded, feature modules implemented, Prisma schema + migration, tests, README, Swagger endpoints.

- Example 2: Incremental feature implementation
  - Input: existing repo + new feature spec + updated wireframe fragment
  - Prompt: "Add the 'create article' feature according to this spec and wireframe fragment. Update Prisma, add Nest module, implement frontend pages and tests."
  - Expected outcome: Minimal changes restricted to feature modules and shared libs, migrations added, tests passing for new ACs.

Best Practices

- Always align strictly with planner AC/QA. If AC contradicts wireframe structure, prioritize AC and flag the contradiction in ISSUES.md.
- Prefer Server Components in Next.js for static content and use client components only for interactive widgets (hydrate responsibly).
- Keep backend modules feature-scoped and independent to support Nx incremental builds and caching.
- Keep database schema minimal and explicit. Add fields only when required by AC; use TODOs and issues for optional/unclear fields.
- Use Zod as single source of truth for validation; derive DTO types from Zod schemas where feasible.
- Write tests to encode ACs; acceptance tests should be BDD-style mapping to AC statements.

Outputs required from the agent when invoked

- A downloadable repository (zip or git diff) with a working Nx + pnpm workspace
- README with setup and run instructions
- Prisma schema (schema.prisma) and at least one migration file
- API documented as OpenAPI/Swagger JSON and integrated into NestJS
- Automated tests (unit, integration, end-to-end) that reflect ACs
- ISSUES.md enumerating unanswered questions and TODO markers in code

Integration and related links

- Nx: https://nx.dev
- Next.js App Router: https://nextjs.org/docs/app
- NestJS + Fastify: https://docs.nestjs.com/techniques/performance#fastify
- Prisma: https://www.prisma.io/docs
- Zod: https://github.com/colinhacks/zod
- Playwright: https://playwright.dev

Examples of prompts to call this Skill

- "Implement the planner spec and wireframe into a full Nx monorepo. Create frontend and backend apps, Prisma schema, migrations, Zod validation, and tests. Mark unknowns with TODO and ISSUES.md."
- "Add feature X from this spec to an existing Nx repo. Keep structure from provided wireframe fragment."

Best output verification steps for a reviewer

1. pnpm install
2. pnpm migrate (or pnpm prisma migrate deploy)
3. pnpm build
4. pnpm test (unit/integration) and pnpm e2e (Playwright)
5. Open Swagger endpoint at /api/docs or configured path to inspect API schema

Notes and default assumptions

- If planner AC lacks DB field types, assume simple scalar types (string, boolean, datetime, integer) and mark the assumption as TODO.
- If wireframe has no explicit routes, derive routes from semantic container ids/classes or file names; list derivation in ISSUES.md.
- If the deliverable format is unspecified, produce a zip and a patch-friendly git-style diff file.

