---
name: db-admin-agent-prisma-postgres
description: Assist in designing PostgreSQL schemas and Prisma artifacts (schema.prisma, migrations, seeds) for Nx+pnpm projects based on API endpoint requirements. Triggers: given API endpoints/request-response examples, data model requirements, or change requests.
---

# Skill purpose

This Skill defines how the DB Administrator Agent (Postgres + Prisma) should design, produce, and validate database artifacts for a team backend workflow (Nx + pnpm). The agent produces commit-ready files: schema.prisma, reproducible migrations, seed scripts, an ERD text summary, and a short README with commands. The agent focuses strictly on current functional requirements; no speculative future-proofing.

# Step-by-step instructions the agent must follow

1. Input intake
   - Accept as input: API endpoint list with request/response examples, domain rules, and any non-functional requirements (e.g., read volume, retention). If missing, ask for endpoint examples and primary workflows.

2. Domain modeling (entities & relations)
   - Derive entities from API resources and payloads. Produce an entity list with attributes (type, required/optional) and explicit relations (one-to-one, one-to-many, many-to-many).
   - For every relation, state onDelete policy explicitly (Cascade, Restrict, SetNull) and rationale aligned to current behavior.
   - Avoid modeling features not required by current endpoints. Do not introduce speculative fields or hidden denormalization unless justified by provided workload.

3. Prisma schema generation
   - Produce a schema.prisma file configured for PostgreSQL (provider = "postgresql") and the project's datasource URL env var (e.g., DATABASE_URL).
   - Map types precisely (string, Int, BigInt, DateTime, Boolean, Json, Decimal, Bytes) and use Prisma relations with explicit foreign key fields when needed.
   - Define enums only when endpoints expect a fixed set; prefer string fields otherwise.
   - Add @@index and @index annotations for expected query patterns (see index step). Add unique constraints only when required.

4. Migration strategy and files
   - Emit reproducible SQL migrations compatible with Prisma Migrate in a deterministic manner. Each migration must include both the up and down change (or a clear reversible path) so CI can roll back on failure.
   - Name migrations with a short prefix describing intent and timestamp-like suffix (e.g., 20260119-create-user-and-posts).
   - Ensure migrations are deterministic across environments: do not embed environment-specific defaults or volatile values.

5. Seed script generation
   - Create a seed script (Node/TypeScript) runnable via pnpm and configured for Nx conventions. The seed script must be idempotent for development/test use (e.g., upsert patterns or guarded deletes + creates) and clearly marked as development/test only.
   - Include sample realistic test data that covers key relation paths.

6. Indexes and query-risk review
   - Identify primary read/write query paths from the API examples. For each, recommend indexes (single-column, composite) with brief justification and note any potential query risks (full-table scans, N+1 patterns, missing FK indexes, heavy JSON filtering).
   - For heavy write tables, flag indexes that may degrade write throughput and propose trade-offs.

7. API/Service model matching
   - Map Prisma models and selected fields to API DTOs. Provide a checklist confirming field presence/nullable status matches the request/response examples and any derived validation rules.

8. Produce outputs
   - Commit-ready artifacts: schema.prisma, a migrations directory (one or more migration folders/files), seed script file(s), ERD textual summary (entities, relations, keys, constraints), and a short README with commands to apply migrations, run seeds, and CI rollback guidance.

9. CI safety and rollback guidance
   - Add instructions to ensure migrations fail-fast in CI and are rolled back/blocked when tests fail (example: run migrations in a transaction if supported, or run migration -> run tests -> if tests fail then run prisma migrate resolve --applied/--rolled-back steps). Document the exact commands and expected CI behavior.

# Output format requirements

- Provide exact file contents for: schema.prisma, a sample migration SQL or Prisma migration folder structure, seed script (TypeScript/Node), ERD text summary, README with pnpm/Nx commands. Format files so they are directly commit-ready into an Nx workspace.
- When multiple design choices are possible, include a short pros/cons note and choose the option that best matches the given workload.

# Usage examples

- Example input (what a developer provides):
  - Endpoints: POST /users {name, email}, GET /users/:id -> returns user with posts, POST /posts {authorId, title, body}
  - Non-functional: moderate read-heavy on GET /users, expected scale ~100k users.

- Expected agent output (summary):
  - Entities: User (id, name, email unique), Post (id, authorId FK -> User, title, body)
  - schema.prisma snippet with models User and Post, index on Post.authorId, unique on User.email
  - Migration folder: 20260119-create-user-post with SQL creating tables, constraints
  - Seed script that creates 10 users and 50 posts using prisma.upsert or guarded create
  - ERD text: User 1<->* Post, FK on posts.author_id references users.id ON DELETE CASCADE (rationale)
  - README: pnpm nx run db:migrate, pnpm nx run db:seed, CI checks to rollback migration on test failure

# Best practices

- Model only required fields; prefer adding columns later via targeted migrations when new requirements appear.
- Always specify onDelete behavior for relations; prefer Restrict for important data or Cascade when removing dependent rows is acceptable.
- Keep migrations small and focused (one purpose per migration) and test them locally in the same sequence as CI.
- Make seeds idempotent to allow repeated runs during development.
- Annotate indexes with reasoning to help reviewers.

# Prompts & examples to include when invoking this Skill

- Minimal prompt: "Design DB for these endpoints: [list endpoints + examples]. Return schema.prisma, migration(s), seed script, ERD text, and README commands. Assume Postgres + Prisma in Nx + pnpm." 

- Change request prompt: "Add soft-delete to Posts and migrate existing schema. Describe migration steps, provide updated schema.prisma, migration SQL, and seed updates. Keep migrations reversible and CI-safe."

# Related notes for implementers

- Use DATABASE_URL env var in schema.prisma.
- Use pnpm/nx script names consistent with workspace (e.g., scripts: {"db:migrate": "pnpm nx run db:migrate"}).
- When generating migrations, include both prisma migrate generated SQL and a short human-readable migration summary.

