---
name: monorepo-stack-setup-guide
description: A developer-facing setup and run guide for an Nx + pnpm monorepo using NestJS (Fastify), Next.js, PostgreSQL + Prisma, Zod, and Jest/Vitest/Playwright. Includes step-by-step install, local run, migration, common commands, and troubleshooting tips for junior and senior devs.
---

# Monorepo Stack Setup & Run Guide

Purpose

This Skill provides a clear, actionable developer guide for setting up, running, testing, and troubleshooting a monorepo built with Nx + pnpm. It targets the whole dev team (including juniors) and focuses on installation and local run steps, with command examples and a troubleshooting checklist. Use this when you need to onboard developers, reproduce local environments, or document common workflows.

Step-by-step instructions (what Claude must return when asked to generate the guide)

1. Provide a concise summary of the tech stack and repository layout assumptions.
2. Present a pre-requisites section listing OS, tools, and versions (Node, pnpm, Nx, Docker optional) with commands to check versions.
3. Provide an ordered, copy-paste-friendly installation and local-run section that includes:
   - Clone and basic repo commands
   - pnpm and Nx bootstrap commands
   - Environment file creation and sample .env entries
   - Database setup with Docker Compose (Postgres) and/or local Postgres instructions
   - Prisma setup: generate, migrate, and seed commands
   - Start backend (NestJS Fastify) and frontend (Next.js) locally with example ports
   - How to run tests (Jest/Vitest) and e2e (Playwright) with example commands
4. Insert a Troubleshooting / FAQ checklist for the most common failures during install and run, each entry containing: symptom, likely cause, and a short fix (commands or config changes).
5. Provide a short section of command reference: grouped commands with one-line descriptions for quick copy-paste.
6. Include usage examples (typical developer workflows) showing step sequences: a) new dev onboarding, b) run feature locally, c) run tests and e2e.
7. Add Best Practices and tips for consistency (pnpm lock, migration workflow, environment parity, running tests in CI/local).
8. When filenames or helper scripts are mentioned, reference them as relative paths (placeholders allowed) and show example contents if brief and helpful.
9. Keep content in English, concise, and copy-paste-friendly; prefer terminal commands and short code snippets.

Detailed Guide

1) Assumptions & repo layout

- Monorepo managed by Nx and pnpm
- Apps: apps/backend (NestJS, Fastify), apps/frontend (Next.js)
- Packages/libs: libs/* (shared types, utils)
- Database schema: prisma/schema.prisma at the repo root or apps/backend/prisma
- Scripts are defined in root package.json and in app package.json files

2) Prerequisites (check and install)

- Node >= 18 (example): node -v
- pnpm >= 8: pnpm -v
- nx cli (optional global): nx --version
- Docker & docker-compose (recommended for Postgres): docker --version

3) Clone and initial bootstrap

- git clone <repo-url>
- cd repo
- pnpm install
- pnpm -w nx migrate -- run if repo provides nx migrations (or follow repo-specific instructions)

4) Environment files

- Copy example envs:
  - cp .env.example .env
  - cp apps/backend/.env.example apps/backend/.env
  - cp apps/frontend/.env.example apps/frontend/.env

- Typical entries (placeholders):
  - DATABASE_URL=postgresql://postgres:postgres@localhost:5432/mydb?schema=public
  - NEXT_PUBLIC_API_URL=http://localhost:3000
  - PORT=3000

5) Database setup (Docker Compose) - recommended

- Create a docker-compose.yml at repo root or use provided one. Example minimal service:
  - version: '3.8'
    services:
      db:
        image: postgres:15
        environment:
          POSTGRES_USER: postgres
          POSTGRES_PASSWORD: postgres
          POSTGRES_DB: mydb
        ports:
          - '5432:5432'
        volumes:
          - db-data:/var/lib/postgresql/data
    volumes:
      db-data: {}

- Start DB: docker compose up -d db
- Verify: pg_isready -h localhost -p 5432 (or docker logs)

Alternative: Use a local Postgres instance and ensure DATABASE_URL matches credentials.

6) Prisma: generate, migrate, seed

- From repo root (adjust path if prisma is in backend app):
  - pnpm prisma generate
  - pnpm prisma migrate dev --name init
  - pnpm prisma db seed

- If prisma CLI scripts are in apps/backend: cd apps/backend && pnpm prisma generate && pnpm prisma migrate dev --name init

7) Bootstrap libs and build cache (Nx)

- pnpm nx run-many --target=build --all (optional to warm build cache)
- For development, run concurrent dev servers with Nx:
  - pnpm nx serve frontend
  - pnpm nx serve backend

Or use workspace root scripts (examples):
  - pnpm dev (runs both frontend and backend concurrently)

8) Start servers

- Backend (NestJS + Fastify):
  - pnpm nx serve backend
  - Default: http://localhost:3000 (adjust by project config)

- Frontend (Next.js):
  - pnpm nx serve frontend
  - Default: http://localhost:4200 or http://localhost:3001 depending on workspace config

9) Tests and E2E

- Unit tests (Jest/Vitest): pnpm nx test <project>
- Run all tests: pnpm nx run-many --target=test --all
- Playwright e2e (example): pnpm nx e2e frontend-e2e --config=playwright.config.ts

10) Troubleshooting checklist (common issues)

- Problem: pnpm install fails with permissions or lockfile errors
  - Cause: stale pnpm-lock.yaml or incompatible pnpm version
  - Fix: pnpm install -w --prefer-offline
  - Fix: Ensure pnpm version matches repo (pnpm env use <version> or install pnpm vX globally)

- Problem: Nx commands not found
  - Cause: nx not installed or not in PATH
  - Fix: Use pnpm nx <command> or install nx globally: pnpm add -g nx

- Problem: DATABASE_URL connection refused
  - Cause: Postgres not running or wrong host/port
  - Fix: docker compose up -d db; check env files; ensure ports are exposed

- Problem: Prisma migrate fails (existing schema mismatch)
  - Cause: DB already has schema with different state
  - Fix: Reset DB during dev: pnpm prisma migrate reset --preview-feature (or drop DB and re-create)

- Problem: TypeScript types not found between packages
  - Cause: libs not built or tsconfig path aliases not configured
  - Fix: pnpm nx build <lib> or run pnpm -w build; check tsconfig.base.json paths

- Problem: Hot reload not working
  - Cause: Next/Nest config or Nx caching
  - Fix: Ensure dev server launched (nx serve) and no build step blocks; disable cache if needed for debugging

11) Command reference (copy-paste)

- git clone <repo-url>
- pnpm install
- cp .env.example .env
- docker compose up -d db
- pnpm prisma generate
- pnpm prisma migrate dev --name init
- pnpm nx serve backend
- pnpm nx serve frontend
- pnpm nx test backend
- pnpm nx e2e frontend-e2e

12) Example workflows

- New developer onboarding (quick)
  1. git clone <repo>
  2. pnpm install
  3. cp .env.example .env && cp apps/backend/.env.example apps/backend/.env
  4. docker compose up -d db
  5. pnpm prisma generate && pnpm prisma migrate dev --name init && pnpm prisma db seed
  6. pnpm nx serve backend & pnpm nx serve frontend

- Implement and run a feature locally
  1. pnpm nx g @nrwl/nest:controller api/feature --project=backend (example)
  2. pnpm nx serve backend
  3. Open frontend and exercise feature, or use curl/postman against backend

- Run tests before PR
  1. pnpm nx test backend
  2. pnpm nx test frontend
  3. pnpm nx e2e frontend-e2e

13) Best practices

- Keep pnpm and Nx versions locked in repository documentation
- Use .env.example to show required env variables; do not commit secrets
- Run prisma migrations and seeds in CI for test DB setup
- Use Nx affected commands in CI to only test/build changed projects
- Prefer reproducible Docker Compose setup for dev parity

Usage examples

- Ask Claude: "Generate a step-by-step onboarding checklist for a new dev in this repository" — Claude should return the onboarding workflow (see section 12) adapted with repo-specific values if provided.
- Ask Claude: "Create a troubleshooting entry for Prisma migrate errors" — Claude should return the symptom/cause/fix triple and suggested commands.

Related files (placeholders referenced above)

- docker-compose.yml (root) - example snippet shown in this doc
- prisma/schema.prisma (root or apps/backend/prisma)
- .env.example (root and apps/*)
- pnpm-workspace.yaml and nx.json (repo root)

Best Practices (short)

- Document exact Node/pnpm versions in README and use engines field in package.json
- Add fast-checks or smoke tests in CI after docker compose startup
- Use Nx caching and affected commands to speed CI

