---
name: objective-c-to-swiftui-migration-guide
description: Provides a step-by-step migration plan, prioritized checklist, and risk flags to convert a small Objective-C iOS app (1–5 screens) to a Swift + SwiftUI project. Triggers: requests to plan or audit an ObjC→SwiftUI migration, ask which files/classes to convert first, or request a migration checklist.
---

# Objective-C to SwiftUI Migration Skill

Purpose

This Skill produces a concise, prioritized migration plan and checklist to convert a small Objective-C iOS project (roughly 1–5 screens, limited Objective-C-only code) into a modern Swift + SwiftUI codebase. It identifies which files/classes to convert first, a safe conversion order, per-item risk flags, and a stepwise execution guide you can follow or present to an engineering team.

Step-by-step instructions (what Claude must do)

1. Confirm project scope and constraints
   - Ask the user for: number of screens, presence of storyboards/xibs, use of UIKit-only features (navigation, custom view controllers), Core Data/third-party frameworks, and any ObjC runtime-heavy code (swizzling, macros).
   - If user already said "small app, 1–5 screens", proceed using that assumption.

2. Analyze architecture and entry points (conceptually, via user descriptions)
   - Request or infer list of top-level UIViewControllers, AppDelegate/SceneDelegate usage, and any shared manager or model classes.
   - Identify UI code vs non-UI code (models, services, networking, data persistence).

3. Produce prioritized conversion order
   - Provide a ranked list of files/classes to convert first. Use this default ordering for small apps unless user-specific constraints change:
     1) Models and business logic (Objective-C classes with no UIKit dependency)
     2) Networking, services, helpers, and managers
     3) App bootstrap (AppDelegate / SceneDelegate) and routing entry points
     4) Simple view controllers whose UI can map directly to SwiftUI views
     5) Complex view controllers with heavy UIKit-specific code (custom transitions, container controllers)
     6) Storyboards/xibs and view hierarchies (replace or extract views into SwiftUI components)
   - For each item, include a short rationale and conversion difficulty (Low/Medium/High).

4. For each file/class in the prioritized list, generate a small, actionable checklist
   - Example checklist items:
     - Review dependencies: does this file import or call UIKit? If not, convert to Swift class or struct.
     - Convert Objective-C model to Swift: map properties, initializers, optionals, and Codable if needed.
     - Replace delegates/callbacks with Swift closures, Combine, or async/await where appropriate.
     - Add unit tests or verify existing tests still pass.
   - Mark risk flags for: Objective-C runtime usage, macros, swizzling, heavy use of categories, and C pointers. Provide mitigation notes.

5. Provide migration script/snippets suggestions (high level)
   - Recommend tools and commands: use Xcode migrate-to-Swift assistance sparingly, use Swift wrappers for ObjC bridging, create mixed-language targets, and gradually replace modules.
   - Suggest sample snippet patterns (how to expose ObjC to Swift with bridging header, how to wrap an ObjC manager in a Swift class).

6. Give a final checklist and acceptance criteria
   - Compile success, UI parity for converted screens, unit tests passing, performance sanity checks, and code review guidelines.

7. Offer follow-ups
   - Ask if the user wants a per-file conversion template, example SwiftUI view translations, or migration scripts.

Usage examples

- User prompt: "Help me migrate a 3-screen Objective-C app with storyboards and a networking manager to SwiftUI. Which files should I convert first and what are key risks?"
  - Expected Skill output: short confirmation of assumptions → prioritized file list (models → networking → AppDelegate → simple VCs → complex VCs → storyboards) → per-item checklist with risk flags.

- User prompt: "I have an ObjC model class and a view controller that uses delegates. Give a conversion checklist for both."
  - Expected Skill output: model conversion checklist (properties → optionals/Codable → tests) and view controller checklist (identify navigation, convert UI to SwiftUI, map delegates to Combine/closures, migration steps).

Best practices

- Migrate incrementally: convert non-UI code first so you can test logic reuse in Swift before reworking UI.
- Keep the app buildable at each step: use mixed ObjC/Swift targets and bridging headers; convert modules one at a time.
- Prefer Swift idioms: use structs for simple models, use Swift concurrency (async/await) where appropriate, and replace delegate patterns with Combine or closures for simpler code.
- Flag and postpone risky conversions (runtime macros, swizzling) until late; consider wrapping such code rather than translating immediately.
- Write tests or verify behavior after each converted unit to reduce regression risk.

Examples of checklists (concise templates)

- Model class (Objective-C → Swift)
  - Confirm no UIKit imports.
  - Map properties and types; mark optionals.
  - Implement Codable if used for networking/persistence.
  - Add unit tests or run existing tests.
  - Replace usage sites to reference new Swift type.

- Simple UIViewController -> SwiftUI View
  - Identify view lifecycle usage (viewDidLoad, viewWillAppear) and map to onAppear or init logic.
  - Extract UI elements into SwiftUI components.
  - Replace IBOutlets/IBActions with SwiftUI bindings and actions.
  - Keep navigation logic: map to NavigationStack/NavigationLink or custom coordinator.
  - Test UI behavior and interactions.

Risk flags and mitigations

- Uses Objective-C runtime features (swizzling, objc_msgSend): HIGH risk — keep as ObjC or wrap in a stable bridging API and document thoroughly.
- Heavy macro usage or C pointers: HIGH risk — translate carefully or isolate in a C/ObjC module and wrap.
- Complex custom UIKit behaviors (container controllers, custom transitions): MEDIUM–HIGH risk — consider keeping in UIKit and hosting with UIViewControllerRepresentable until rewritten.

When to ask follow-up questions

- If the user hasn't provided file lists, ask for at least the top-level classes or describe screens and major managers.
- If third-party frameworks or Core Data are present, ask which ones and whether replacement is planned.

Related notes and links

- Recommend official Apple resources: Swift migration guide, SwiftUI documentation, and Apple sample code for UIViewControllerRepresentable and App lifecycle migration.

