---
name: unreal-cpp-refactor-suggestions
description: Provides conservative, behavior-preserving automated refactoring suggestions for Unreal Engine C++ code focused on error handling, extensibility, and readability. Triggers: paste/source file, request for refactor, or code review comments.
---

Skill purpose

This Skill analyzes Unreal Engine C++ code and generates conservative, actionable refactoring suggestions that improve error handling, extensibility, and readability while preserving existing behavior. It outputs a clear refactor summary, step-by-step rationale for each change, and a suggested patch or diff plus test/review guidance.

Step-by-step instructions Claude must follow

1. Ingest input
   - Accept either a code snippet, one or more source/header files, or a brief description of the problematic behavior and project conventions (e.g., coding standards, engine version).
   - If multiple files are provided, consider cross-file interactions (headers, UPROPERTY, UFUNCTION, module boundaries).

2. Quick static analysis
   - Identify obvious issues: unchecked return values, missing nullptr checks, raw pointer ownership ambiguity, duplicated logic, large functions, long parameter lists, public data members, fragile macro usage, risky casts (reinterpret_cast/C-style), inefficient loops, excessive header includes, and missing const-correctness.
   - Note Unreal-specific concerns: UObject lifetimes, replication considerations, Garbage Collection markers, RPC validation, correct use of FName/FString, thread-safety with Async/Task Graph, and proper use of GENERATED_BODY/REFLECT macros.

3. Classify findings by severity and conservativeness
   - Class A (high priority, safe): definite bugs or undefined behavior (null deref, use-after-free, missing validity checks for UObjects). Recommend immediate fixes.
   - Class B (medium priority, safe): improvements that preserve behavior (extract function, remove duplication, replace raw loops with range-based loops, add const, use TOptional/TSharedPtr where appropriate), flagged as conservative refactors.
   - Class C (low risk, optional): design-level enhancements that may require broader review (introduce interfaces, add dependency injection, change ownership semantics). Provide as optional suggestions with migration notes.

4. Produce refactor suggestions
   - For each finding, produce:
     - A concise title (one line)
     - A short description of the problem
     - The recommended change (code-level) and why it preserves behavior
     - A suggested code patch or diff in unified format (context lines + changed lines). Keep patches minimal and focused.
     - Risk assessment (behavior-preserving? compile-only risk? needs runtime test?) and required follow-ups (update headers, adjust build.cs, add include, regenerate reflection if necessary).

5. Explain step-by-step rationale
   - For each suggested change, give a numbered explanation: what to change, why, how it helps error handling/extensibility/readability, and which Unreal-specific pitfalls it avoids.

6. Provide tests and review guidance
   - Suggest unit/integration tests, manual steps to validate (in-editor checks, PIE, dedicated test maps), and places to watch for regressions (replication, GC, editor-only initializations).

7. Provide a short summary and prioritized checklist
   - One-paragraph summary of the most important changes
   - A checklist (apply patch, run compile, run tests, manual playtest, code review)

8. Offer optional automated patches
   - If requested and enough context is provided, output full patches for files. Otherwise, provide minimal diffs and exact replacement snippets.

Usage examples

Example 1 — Small function with null checks and duplication
Input: a 40-line Actor method that dereferences a component pointer without checks and duplicates a transform calculation.
Output: (a) Two Class A suggestions: add IsValid/ensure/UE_LOG and early return; (b) Class B suggestion: extract transform calculation into a helper function; (c) Unified diff for the function showing added checks and extracted helper; (d) step-by-step rationale and test instructions (play in PIE and check logs).

Example 2 — UObject ownership ambiguity
Input: a factory function returning raw pointers and storing them in non-UPROPERTY fields.
Output: Recommend converting to NewObject with proper Outer and UPROPERTY storage or using TStrongObjectPtr/TWeakObjectPtr where appropriate. Provide migration patch, note need to re-run editor and consider replication impacts, list tests to validate lifetime.

Best practices

- Be conservative: prefer changes that preserve runtime behavior and require minimal API changes.
- Favor explicit checks (IsValid/ensure) over silent failures.
- Prefer engine types (FName, TOptional, TArray, TSharedPtr, TWeakObjectPtr) to raw primitives where ownership/intent matters.
- Keep diffs small and focused; explain each change clearly so reviewers can approve quickly.
- When refactoring for extensibility, prefer interface (I*) abstractions only when multiple implementations exist; otherwise prefer small helper functions or protected virtuals.
- Always call out reflection/regeneration needs (e.g., adding UPROPERTY or UFUNCTION that affect serialization or replication).

When to ask for more context

- If code depends on project-specific macros, build settings, or engine modifications, request the relevant build.cs or macro definitions.
- If lifetime/ownership is unclear, ask for call sites or higher-level usage examples.
- If the user wants automated patches applied, request the exact file paths and confirm permission to modify those files.

Related links and references

- Unreal C++ coding standards: https://docs.unrealengine.com/
- Recommended APIs: IsValid, TSharedPtr/TWeakPtr, NewObject, UPROPERTY, FScopedSlowTask

Examples and templates

- Provide diffs in unified format and single-file replacement snippets for copy-paste. If the user supplies file paths and grants permission, offer a patch bundle (unified diff) ready to apply with patch/git.

