---
name: cpp-compile-error-fixer
description: Provides focused diagnosis and actionable fixes for C++ compilation errors produced by g++ or clang++. Triggered by providing compiler command, full error output, and relevant source snippets or filenames.
---

# C++ Compile Error Fixer

Skill purpose

This Skill helps developers quickly diagnose and fix C++ compile-time errors from g++/clang++. Given the compiler invocation, the complete error output, and the relevant source snippets or filenames (with line numbers when possible), the Skill identifies the likely root cause, presents several concrete fix options, and provides small targeted code examples or patches that can be applied to specific lines.

Step-by-step instructions Claude must follow

1. Prompt the user (if not provided) to supply:
   - The exact compiler command used (e.g., g++ -std=c++17 -Wall -O2 main.cpp)
   - The full compiler error/warning output (copy-paste entire terminal output)
   - The relevant source file(s) or snippets that contain the error locations, including file names and line numbers if available
   - The target compiler (g++ or clang++) and version if known
2. Parse the compiler output to extract:
   - Each distinct error/warning message
   - File name(s) and line number(s) associated with each message
   - The primary error (first fatal error) and related secondary notes
3. For each primary error, classify it into common categories (examples):
   - Syntax error (missing semicolon, unmatched braces)
   - Type mismatch or template substitution failure
   - Missing include or undefined symbol
   - Linker error vs compiler error (distinguish and handle separately)
   - Language standard mismatch (features not enabled)
   - ODR / multiple definition problems
4. For each error, produce a compact, structured response containing:
   - One-line summary of the likely root cause
   - The exact file and line(s) to change (use line numbers from the error output; if missing, indicate approximate location and request snippet)
   - At least two alternative fixes (short, prioritized):
     a. Minimal fix (small code change or include) with a short code snippet showing only the changed lines or a small contiguous patch
     b. Robust fix (safer or more idiomatic approach) with a short example
     c. (Optional) Workaround or compiler-flag solution if applicable
   - A suggested compile command to verify the fix
   - A short note about potential side effects or tradeoffs for each fix
5. When proposing code snippets, include only the minimal context needed (3–10 lines) and indicate which line(s) to change. Prefer showing +- style minimal patch or an explicit edited snippet with line numbers when possible.
6. If the cause is ambiguous or more context is required, list the specific additional information required (e.g., surrounding 10 lines of code, header files, template instantiation context, build system details).
7. For linker errors, list required object files or libraries to link and provide exact linker flag suggestions (e.g., -l, -L, or order changes), and mention common pitfalls (missing -pthread, order of libs).
8. For template/long diagnostics, extract the concise user-readable summary and avoid repeating full compiler backtraces unless needed; include the important template instantiation lines with context.
9. Always include a brief compile-and-test command users can run to validate the proposed fix (e.g., g++ -std=c++17 -Wall main.cpp -o main).
10. Keep responses concise, precise, and actionable. When multiple errors exist, prioritize the first fatal compile error and note remaining errors may change after the first fix.

Usage examples

Example 1 — simple syntax error

User provides:
- Command: g++ -std=c++17 -Wall main.cpp
- Error output: main.cpp:12:5: error: expected ';' after expression
- Source snippet: lines 8-14 of main.cpp

Desired response:
- One-line cause: missing semicolon at end of statement on line 12
- Minimal fix: add ';' at end of line 12 (show the line before and after)
- Robust fix: ensure the preceding block is closed properly; show corrected snippet
- Compile command to test

Example 2 — undefined reference (linker)

User provides:
- Command: g++ -std=c++17 main.o helper.o -o app
- Error: undefined reference to `helper::doWork()`

Desired response:
- Identify linker error, likely missing object file or wrong symbol mangling (extern "C" vs C++)
- Suggest ensuring helper.cpp is compiled and linked, or correct function signature / extern linkage
- Provide exact commands to compile and link helper.cpp

Example 3 — template instantiation error

User provides long template error from g++ with note about no matching function for call to foo(Bar)

Desired response:
- Extract and summarize the concrete cause (e.g., overload resolution fails because candidate expects const T&)
- Show minimal edited template or call-site fix (e.g., add conversion, change template parameter, or add overload)
- Provide small code snippet for each option and note tradeoffs

Best practices

- Always request the full original compiler output and the minimal reproducible snippet if the error context is insufficient.
- Prefer targeted, minimal changes over large rewrites; provide alternative options when the minimal change is risky.
- When suggesting compiler flags as a fix (e.g., -std=c++17), explain the targeted feature and compatibility risk.
- For ambiguous diagnostics, explicitly list which additional lines or files are needed and why.
- Keep the first response focused on the first fatal error; offer to iteratively fix further errors after the first is resolved.

When to ask for more information

- Missing file/line references in compiler output
- Long template instantiation chains where the relevant instantiation site is not shown
- Linker errors when build/link command is not provided
- Errors that depend on macro definitions or conditional compilation

Related links and commands

- Common compile/test commands to include in responses:
  - g++ -std=c++17 -Wall source.cpp -o program
  - clang++ -std=c++20 -Wall source.cpp -o program
  - g++ -c file.cpp && g++ file.o -o app

