#!/usr/bin/env bash
# Runner script: runs Playwright tests and writes a JSON log summary.

LOG_DIR=${LOG_DIR:-./test-logs}
mkdir -p "$LOG_DIR"
TS=$(date -u +"%Y-%m-%dT%H-%M-%SZ")
LOG_FILE="$LOG_DIR/test-result-$TS.json"

# Run Playwright tests and capture output. Adjust command if using a custom runner.
# Requires: npm install -D @playwright/test

npx playwright test --reporter=json > "$LOG_FILE" 2>&1
EXIT_CODE=$?

# Parse concise summary from JSON report if available
if [ -f "$LOG_FILE" ]; then
  # Extract totals and failed test names using node for portability
  node - <<'NODE'
const fs = require('fs');
const path = process.argv[1];
const f = fs.readFileSync(process.argv[2], 'utf8');
try {
  const j = JSON.parse(f);
  const stats = j.suites ? { total: j.total || 0, passed: j.passed || 0, failed: j.failed || 0 } : {};
  // Fallback parsing for reporter json array
  if (Array.isArray(j)) {
    const results = j.filter(r => r.type === 'test');
    const total = results.length;
    const failed = results.filter(r => r.result && r.result.status === 'failed');
    console.log(JSON.stringify({ total, passed: total - failed.length, failed: failed.map(f => f.title) }));
  } else {
    console.log(JSON.stringify(stats));
  }
} catch (e) {
  console.log(JSON.stringify({ raw: true }));
}
NODE
  echo "Detailed log: $LOG_FILE"
else
  echo "No log file created. Exit code: $EXIT_CODE"
fi

exit $EXIT_CODE
