---
name: js-error-fixer
description: Analyze JavaScript console errors and stack traces (from browser DevTools), locate the offending code, and produce a ready-to-apply unified diff patch with a simple smoke/unit test, run command, and risk notes. Common triggers: paste of console message + stack trace, repository file listing or file contents, or request to "just fix this error." 
allowed-tools:

---

# JavaScript Error Fixer Skill

Purpose

This Skill helps frontend developers quickly diagnose and fix JavaScript runtime errors observed in the browser DevTools console. Given a console error message and stack trace (and optionally repository or file contents), the Skill locates the likely faulty code, produces a minimal, well-commented fix, generates a simple smoke or unit test to validate the fix, and returns a unified diff/patch that can be applied with git. It also includes the command(s) to run the test and short notes about potential risks and why the change is safe.

Step-by-step instructions (what Claude must do when this Skill is invoked)

1. Parse input
   - Extract the console error message, error name/type, and full stack trace as provided.
   - If additional context is provided (file contents, repository tree, snippet of relevant files, or a file path pattern), record those references.

2. Map stack frames to source files
   - Normalize stack frames to project paths if sourcemaps are referenced. If sourcemaps are not available and the frame lists transpiled/bundled files, ask user for the original source or sourcemap access.
   - If a frame contains a URL with line/column, use that to identify the file and approximate location.

3. Locate the offending code
   - Open the referenced file(s) or request file contents if not supplied. If multiple frames point to different call sites, focus on the topmost project-frame (first non-third-party frame) that is inside the user's codebase.
   - Identify likely bug patterns matching the error type (e.g., TypeError: cannot read property 'x' of undefined → missing null check or incorrect assumption about value shape; ReferenceError → misspelling or missing import; RangeError/URIError/etc. → invalid inputs).

4. Produce a minimal, correct code change
   - Create a minimal change that fixes the immediate error and keeps behavior as intended. Prefer explicit checks, early returns, or small refactors rather than large rewrites.
   - Add brief inline comments explaining the intent of the change and why it prevents the error.
   - Ensure coding style is consistent with surrounding code (reasonable defaults if style info absent).

5. Create a smoke/unit test
   - Add a small test (Jest, Mocha+Chai, or the project's existing test framework if provided) that reproduces the failing condition and verifies the fix avoids the crash and preserves expected output.
   - Provide a one-line command to run just that test (e.g., npm test -- path/to/test or npx jest path/to/test).

6. Generate a unified diff/patch
   - Format changes as a unified git diff patch (git-style) showing exactly which files and lines changed.
   - Also include the new test file or updated test in the diff.

7. Risk analysis and notes
   - Provide 3–6 concise bullet points: what the change fixes, any behavior changes, potential edge cases, performance considerations, and recommendations for further review (e.g., additional integration tests or places to audit).

8. If insufficient information
   - If necessary files, sourcemaps, or repo context are missing, produce a short list of precisely what is needed (file path(s), file contents around lines noted, package/test runner details, or access to sourcemaps). Offer a best-effort patch using reasonable placeholders and clearly mark them.

Usage examples

Example 1 — Simple null-check fix

Input provided by user:
- console: TypeError: Cannot read property 'length' of undefined
- stack: at renderList (src/components/List.jsx:42:13) -> src/components/List.jsx shown to user

What the Skill returns:
- Unified diff that modifies src/components/List.jsx to add a guard: if (!items) return null; with comment
- Adds a Jest test src/components/__tests__/List.test.js that calls render with undefined items and expects no crash
- Run command: npx jest src/components/__tests__/List.test.js
- Risk notes: explains how this prevents crash, potential UI change (renders nothing), and suggestion to audit upstream callers for why items can be undefined.

Example 2 — Missing import / wrong variable name

Input:
- ReferenceError: myFunc is not defined
- Stack: at handleClick (src/pages/Home.js:88:5)

What the Skill returns:
- Unified diff that either corrects the misspelled import or re-exports/defines myFunc, with comment noting the root cause
- Simple unit test that invokes handleClick and verifies myFunc is invoked (or that the handler executes without throwing)
- Test run command and risk notes.

Best practices for users

- Provide the full console message and full stack trace (not just the top frame). Include source-mapped frames if available.
- If possible, supply the relevant file contents or a small repo snapshot (or grant repo access) so the Skill can produce exact diffs. If the project uses TypeScript, include compiled maps or original .ts/.tsx files.
- Tell the Skill the test framework in use (Jest, Mocha, etc.) or include package.json when possible.
- If you want a more aggressive refactor instead of a minimal fix, state that explicitly.
- Review the patch before applying; the Skill focuses on small, local, low-risk fixes but may make assumptions without full context.

When to ask follow-up questions

- When stack traces point only to minified/bundled code and sourcemaps are not provided.
- When multiple equally likely frames exist and additional runtime context is needed to choose the right file.
- When the project uses an uncommon build/test setup and the test command cannot be inferred.

Placeholders and defaults the Skill will use when details are missing

- Default test framework: Jest (create a test file using jest.fn and expect). Provide commands using npx jest.
- Default patch format: unified git diff (patch compatible with git apply / git am).
- Default coding style: match surrounding whitespace/semicolon usage when file provided; otherwise use standard 2-space indentation for JS and semicolons omitted where unclear.

Links to related files or scripts

- None embedded. If the user supplies repository files, reference them in the diff outputs and in recommendations.

Examples of prompts to invoke this Skill (how users should call it)

- "Fix this error: TypeError: Cannot read property 'map' of undefined\nSTACK: ..." + attach src/components/Thing.jsx
- "Just fix this ReferenceError. Here's the stack trace and Home.js contents." 
- "I want a patch I can apply with git. Error and stack trace below, project uses Jest."

Notes for the assistant implementer

- Always return the unified diff as plain text in the primary response, along with the test run command and risk bullets.
- If producing a best-effort patch with placeholders, clearly annotate placeholder lines in the patch and in a short header comment.
- Keep patches minimal: change only the files needed to fix the error and add tests.

