---
name: my-skill-1771046015295
description: Automatically crawl LME (non-ferrous metals) prices from Naver Finance and save to Google Sheets / Drive using Google Apps Script
---

Skill purpose

This Skill provides a Google Apps Script (GAS) implementation to automatically crawl non-ferrous metal (LME) prices from Naver Finance (finance.naver.com) and save the extracted data into a Google Sheet or daily spreadsheet files in a Google Drive folder. It includes working example scripts, a robust parsing implementation for the Naver page (EUC-KR), setup steps, scheduling instructions, usage examples, and best practices so the Skill can be installed and run reliably.

Step-by-step instructions (for Claude to follow when generating or updating user code)

1. Provide the user with the GAS script files or paste-ready code blocks. Ensure encoding handling (EUC-KR) when fetching the Naver page using UrlFetchApp.getContentText("EUC-KR").
2. Use a reliable parsing method suitable for GAS (regular expressions operating on the HTML string). Implement a parseLMEData(html) function that returns an array of rows: [timestamp, metalName, priceUSDPerTon, change, percentChange].
3. Implement two main example functions:
   - fetchLMEPrices(): fetch and append to the active spreadsheet sheet named "LME_prices".
   - fetchAndSaveLMEToDrive(): fetch and save into a daily-named spreadsheet in a specified Drive folder.
4. Ensure the script checks and creates the Drive folder when needed and creates or reuses a spreadsheet file with the daily name.
5. Add header rows and simple styling for readability.
6. Add logging and basic error handling (muteHttpExceptions, check response code).
7. Provide clear user instructions for permissions, setting the folder name, and adding a time-driven trigger.
8. Provide usage examples and best practices in English.

Working GAS code examples

Example 1 — Append to active spreadsheet (sheet name: "LME_prices")

function fetchLMEPrices() {
  const url = 'https://finance.naver.com/marketindex/nonferrous.naver';
  const response = UrlFetchApp.fetch(url, {muteHttpExceptions: true});
  if (response.getResponseCode() !== 200) {
    Logger.log('Fetch failed: ' + response.getResponseCode());
    return;
  }
  const html = response.getContentText('EUC-KR');
  const data = parseLMEData(html);
  if (!data || data.length === 0) {
    Logger.log('No data parsed');
    return;
  }

  const ss = SpreadsheetApp.getActiveSpreadsheet();
  const sheetName = 'LME_prices';
  const sheet = ss.getSheetByName(sheetName) || ss.insertSheet(sheetName);

  if (sheet.getLastRow() === 0) {
    sheet.appendRow(['Collected At', 'Metal', 'Price (USD/ton)', 'Change', 'Change (%)']);
  }
  data.forEach(row => sheet.appendRow(row));
  Logger.log('Saved ' + data.length + ' rows to sheet: ' + sheetName);
}

Example 2 — Save to a daily spreadsheet file in a Drive folder

function fetchAndSaveLMEToDrive() {
  const FOLDER_NAME = 'LME_prices_data'; // Change this to your preferred folder name
  const today = Utilities.formatDate(new Date(), 'Asia/Seoul', 'yyyy-MM-dd');
  const fileName = 'LME_prices_' + today;

  const url = 'https://finance.naver.com/marketindex/nonferrous.naver';
  const response = UrlFetchApp.fetch(url, {muteHttpExceptions: true});
  if (response.getResponseCode() !== 200) {
    Logger.log('Fetch failed: ' + response.getResponseCode());
    return;
  }
  const html = response.getContentText('EUC-KR');
  const data = parseLMEData(html);
  if (!data || data.length === 0) {
    Logger.log('No data parsed');
    return;
  }

  // Find or create folder
  const folders = DriveApp.getFoldersByName(FOLDER_NAME);
  const folder = folders.hasNext() ? folders.next() : DriveApp.createFolder(FOLDER_NAME);

  // Find or create file for today
  const files = folder.getFilesByName(fileName);
  let spreadsheet;
  if (files.hasNext()) {
    spreadsheet = SpreadsheetApp.open(files.next());
  } else {
    spreadsheet = SpreadsheetApp.create(fileName);
    // Move created file into the desired folder
    DriveApp.getFileById(spreadsheet.getId()).moveTo(folder);
  }

  const sheet = spreadsheet.getSheets()[0];
  if (sheet.getLastRow() === 0) {
    sheet.appendRow(['Collected At', 'Metal', 'Price (USD/ton)', 'Change', 'Change (%)']);
    // Light header formatting
    sheet.getRange(1, 1, 1, 5).setFontWeight('bold').setBackground('#4a86e8').setFontColor('#ffffff');
  }
  data.forEach(row => sheet.appendRow(row));
  Logger.log('[Done] ' + fileName + ' saved into folder: ' + FOLDER_NAME + ' (' + data.length + ' rows)');
}

Parsing implementation

function parseLMEData(html) {
  // Returns array of rows: [timestamp (string), metalName, price, change, percentChange]
  const results = [];
  const timestamp = Utilities.formatDate(new Date(), 'Asia/Seoul', 'yyyy-MM-dd HH:mm:ss');

  // Naver marketindex nonferrous page often contains items in blocks with class or td structure.
  // This regex-based parser targets rows with a metal title and adjacent numeric columns.
  // It is tolerant to small HTML changes but may require update if Naver changes layout.

  // Find table-like blocks that include a title and numeric values
  const rowRegex = /<td class="tit">\s*<a[^>]*>([^<]+)<\/a>\s*<\/td>[\s\S]*?<td class="num">\s*([^<\n\r]+?)\s*<\/td>[\s\S]*?<td[^>]*>\s*([^<\n\r]+?)\s*<\/td>/g;
  let m;
  while ((m = rowRegex.exec(html)) !== null) {
    try {
      const metalName = m[1].trim();
      // normalize numbers: remove commas and non-numeric chars except . and -
      const priceRaw = m[2].trim();
      const changeRaw = m[3].trim();

      // Attempt to extract percent change if present in next sibling or within changeRaw
      // Look ahead for a percent cell after the match (simple search)
      const pctRegex = /([+-]?[0-9.,]+\s*%)/;
      let pctMatch = null;
      const nextSlice = html.slice(rowRegex.lastIndex, rowRegex.lastIndex + 200);
      pctMatch = pctRegex.exec(nextSlice);
      const percentChange = pctMatch ? pctMatch[1].replace(/\s+/g, '') : '';

      const clean = s => s.replace(/[,\s]+/g, '').replace(/[^0-9.\-]/g, '');
      const price = clean(priceRaw);
      // change may include + or - and comma
      const change = clean(changeRaw) || '';

      results.push([timestamp, metalName, price, change, percentChange]);
    } catch (e) {
      // continue on parse errors
      Logger.log('Parse error: ' + e);
    }
  }

  // If no rows found, attempt alternative simple parse for common metal names
  if (results.length === 0) {
    const metals = ['구리', '알루미늄', '아연', '납', '주석', '니켈'];
    for (let i = 0; i < metals.length; i++) {
      const name = metals[i];
      const altRegex = new RegExp(name + '[\s\S]{0,120}?class="num">\s*([^<]+)', 'g');
      const alt = altRegex.exec(html);
      if (alt && alt[1]) {
        const price = alt[1].replace(/[,\s]+/g, '').replace(/[^0-9.\-]/g, '');
        results.push([timestamp, name, price, '', '']);
      }
    }
  }

  return results;
}

Usage examples

- Manual run: Open the GAS editor for your spreadsheet and run fetchLMEPrices() to append the latest values to the active sheet named LME_prices.
- Daily file: Run fetchAndSaveLMEToDrive() once, or schedule it using a time-driven trigger to create/append a daily file in the Drive folder named by FOLDER_NAME.

Setting up permissions

When you run the script for the first time you'll be asked to grant permissions. Required scopes (granted by Google UI) include:
- View and manage your spreadsheets in Google Drive (SpreadsheetApp)
- View and manage the files in your Google Drive (DriveApp)
- Connect to an external service (UrlFetchApp)

Make sure to run an initial manual execution and accept permissions before relying on triggers.

How to install and schedule (brief)

1. In Google Sheets: Extensions → Apps Script.
2. Paste the code examples above into Code.gs and save.
3. Optionally change FOLDER_NAME in fetchAndSaveLMEToDrive() to your preferred folder.
4. Run a function manually once to authorize.
5. To schedule automatic runs: In the Apps Script editor, go to Triggers (left menu clock icon) → Add Trigger. Choose the function (fetchAndSaveLMEToDrive), event source: Time-driven, type: Day timer (or Hours timer) and set the preferred time.

Best practices

- Test parsing manually after first deploy. Naver may change page layout; update parseLMEData() accordingly.
- Add exponential retry/backoff if you expect transient network failures.
- Keep a backup copy of important spreadsheets and consider appending to a single master file to simplify analysis.
- Respect scraping policies and avoid excessive request frequency. Use a reasonable schedule (e.g., once per hour or once per day) to reduce load.
- Log failures and notify (e.g., sendEmail) if critical.

Links and references

- Naver Finance non-ferrous page: https://finance.naver.com/marketindex/nonferrous.naver
- Google Apps Script documentation: https://developers.google.com/apps-script

